
#ifndef SPECTROVISSDK_INTERFACE_H
#define SPECTROVISSDK_INTERFACE_H

#ifdef __cplusplus
	#define SPECTROVISSDK_INTERFACE_DECL extern "C"
#endif

#if defined TARGET_OS_WIN

	typedef	long	 SpecStatus;		// Windows Error codes
	typedef	void 	 SpecDevice;		// Windows Device HANDLE = (void*). Since our API 
									// uses SpecDevice*, we use void here, not HANDLE
	#define kSpec_StatusInvalidDevice		ERROR_INVALID_HANDLE // To Windows error code
	#define kSpec_StatusTooManyDevices		ERROR_TOO_MANY_OPEN_FILES
	#define kSpec_StatusDeviceOffline		ERROR_DEVICE_NOT_CONNECTED
	#define kSpec_StatusDevAlreadyInited	ERROR_ALREADY_ASSIGNED 
	#define kSpec_StatusOutOfMemory			ERROR_NOT_ENOUGH_MEMORY
	#define kSpec_StatusInvalidParam		ERROR_INVALID_PARAMETER
	#define kSpec_StatusMutexTimeout		ERROR_TIMEOUT

		
typedef unsigned int UInt32;
typedef int	SInt32;

#elif defined TARGET_OS_MAC

	#include "VST_USB.h"

	typedef SInt32              OSStatus;

	typedef	OSStatus			SpecStatus;
	typedef	VST_USBBulkDevice 	SpecDevice;
	typedef unsigned char		UCHAR;

	#define kSpec_StatusInvalidDevice		(-1)
	#define kSpec_StatusTooManyDevices		(-2)
	#define kSpec_StatusDeviceOffline		(-3)
	#define kSpec_StatusDevAlreadyInited	(-4)
	#define kSpec_StatusOutOfMemory			(-5)
	#define kSpec_StatusInvalidParam		(-6)
	#define kSpec_StatusMutexTimeout		(-7)
	#define kSpec_StatusReadStarved			(-8)

#else
	#error("TARGET_OS_* not defined")
#endif

typedef unsigned short  UInt16;
typedef short	SInt16;



typedef SInt32                          OSStatus;

#define SPECTROVIS_DEVICETYPE   0


typedef enum {
	kSpec_LampNone,
	kSpec_LampNormal,
	kSpec_LampFluorescence1,
	kSpec_LampFluorescence2
} eSpec_Lamp;

///////////////////////////////////////////////////////////////
//
// Publicly available APIs common to all platforms
// 


////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION: Spec_GetDevices										(Public)
//
// Inventory All Spectrometers.
// Return a list of USB Spectrometer Devices currently present, that fit our criteria
// This function is called from GVSpecOpenAllSpectrometers() when the upper layer application
// calls GetAvailableDevices(). This occurs on App startup, when USB devices are plugged in
// and -- on Windows only -- when USB devices are UNPLUGGED. [MAC Unplug events are handled differently]
//
// INPUT:   deviceType   0 for SpectroVis
//          outDevices   Ptr to an array of device pointers
//          howManyDevs  Caller specifies max size of the outDevices array
//     
// RETURN:  outDevices   Array is filled in with device pointers
//          howManyDevs  Has the actual number of valid devices returned
////////////////////////////////////////////////////////////////////////////////
SPECTROVISSDK_INTERFACE_DECL SpecStatus Spec_GetDevices(int DeviceType, SpecDevice** outDevices, int* maxDevices);



//////////////////////////////////////////////////////////////////////
//
// FUNCTION: Spec_GetDevice
// Returns the first spectrometer found  NULL if no device is present.

SPECTROVISSDK_INTERFACE_DECL SpecDevice* Spec_GetDevice(int DeviceType);



/////////////////////////////////////////////////////////////////////////////
//
// FUNCTION:  Spec_InitializeDevice								(Public)
//
//	The calling application MUST call Spec_InitializeDevice for each device to be used.
//  This results in an "Initialize" command being sent to the device which 
//  RESETS all settings to their defaults! 
//
//  In addition, we here read in and STORE device information which will be
//  used in many other calls -- specifically an EndPoint-to-PipeNumber MAP,
//  at this time.  Vendor Specific calls won't work until until this is done.
//
/////////////////////////////////////////////////////////////////////////////
SPECTROVISSDK_INTERFACE_DECL SpecStatus Spec_InitializeDevice(SpecDevice* device);

//////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION: Spec_TerminateDevice										(Public)
//
// Performs Platform-independent cleanup of our device array, 
// then calls OS-specific function to close device.
/////////////////////////////////////////////////////////////////////////////////
SPECTROVISSDK_INTERFACE_DECL SpecStatus Spec_TerminateDevice(SpecDevice* device);

//////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION: 			Spec_TerminateAllDevices				(Public)
//
// Performs Platform-independent cleanup of all devices in the array, 
/////////////////////////////////////////////////////////////////////////////////
SPECTROVISSDK_INTERFACE_DECL SpecStatus Spec_TerminateAllDevices();

//////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION: Spec_GetSpectra											(Public)
//
// --It now gets 
// all the scans, handles dead pixels, pixel-collapsing, boxCar-ing, and 
// scan averaging. 
//
/////////////////////////////////////////////////////////////////////////////////
SPECTROVISSDK_INTERFACE_DECL SpecStatus Spec_GetSpectra(SpecDevice* device, double* spectra, size_t maxSpectraSize);

//////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION: Spec_GetDarkSpectra											(Public)
//
// This Platform-independent function now replaces much of what was in 
// GVSpecGetScan() -- and should be called from GVSpecGetScan(). It now gets 
// all the scans, handles dead pixels, pixel-collapsing, boxCar-ing, and 
// scan averaging. It calls one platform-dependent function: Spec_OSGetSpectra()
//
/////////////////////////////////////////////////////////////////////////////////
SPECTROVISSDK_INTERFACE_DECL SpecStatus Spec_GetDarkSpectra(SpecDevice* device, double* spectra, size_t maxSpectraSize);


////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION: Spec_GetProductID										(Public)
//
// Return the USB product ID for this device type
// We ASSUME here that the info was ALREADY read from the device ON discovery.
////////////////////////////////////////////////////////////////////////////////
SPECTROVISSDK_INTERFACE_DECL SpecStatus Spec_GetProductID(SpecDevice* device, SInt16* outID);

////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION: Spec_GetDeviceName										(Public)
//
// Report the device's product name string from its descriptor.
// We ASSUME here that the info was ALREADY read from the device ON discovery.
////////////////////////////////////////////////////////////////////////////////
SPECTROVISSDK_INTERFACE_DECL SpecStatus Spec_GetDeviceName(SpecDevice* device, char* outNameBuf, size_t bufferSize);

////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION: Spec_GetSerialNumber									(Public)
//
// Report the device's serial number string from its descriptor.
// If we have not read it before, then access the device to do so.
// 
////////////////////////////////////////////////////////////////////////////////
SPECTROVISSDK_INTERFACE_DECL SpecStatus Spec_GetSerialNumber(SpecDevice* device, char* outSNBuf, size_t bufferSize);

////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION: Spec_GetFirmwareVersion									(Public)
//
// Report the device's firmware version string from its descriptor.
// We ASSUME here that the info was ALREADY read from the device ON discovery.
////////////////////////////////////////////////////////////////////////////////
SPECTROVISSDK_INTERFACE_DECL SpecStatus Spec_GetFirmwareVersion(SpecDevice* device, char* outVerBuf, size_t buffersize);


//////////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION: Spec_GetWavelengths											(Public)
//
// Get the mapping from pixel index to wavelengths (in nanometers)
//////////////////////////////////////////////////////////////////////////////////////
SPECTROVISSDK_INTERFACE_DECL SpecStatus Spec_GetWavelengths(SpecDevice* device, double* waveLengthArray, size_t wavelengthArraySize);

////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION: Spec_SetStrobeEnable										(Public)
//
// Turn the specified lamp on or turn all the lamps off.
////////////////////////////////////////////////////////////////////////////////
SPECTROVISSDK_INTERFACE_DECL SpecStatus Spec_SetStrobeEnable(SpecDevice* device, eSpec_Lamp lampType);

////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION: Spec_CalibrationModeChanged							(Public)
//
// will change the set/lights and gain based on the mode of the device (spec specific
//
////////////////////////////////////////////////////////////////////////////////
SPECTROVISSDK_INTERFACE_DECL SpecStatus Spec_CalibrationModeChanged(SpecDevice* device, eSpec_Lamp lampType);

//////////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION: Spec_GetIsUV_VIS											(Public)
//
// Return true if the wavelength range of the device is for a UV Spectrometer
// Exposed public for upper layers... Not used locally
// 
SPECTROVISSDK_INTERFACE_DECL bool	   Spec_GetIsUV_VIS(SpecDevice* device);

/////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION: Spec_SetScansToAverage									(Public)
//
// Set the number of spectra scans to average before returning a result.
////////////////////////////////////////////////////////////////////////////////
SPECTROVISSDK_INTERFACE_DECL void Spec_SetScansToAverage(SpecDevice* device, int scans);

/////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION: Spec_GetScansToAverage									(Public)
//
// Get the number of spectra scans to average before returning a result.
////////////////////////////////////////////////////////////////////////////////
SPECTROVISSDK_INTERFACE_DECL int  Spec_GetScansToAverage(SpecDevice* device);

////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION: Spec_SetBoxcarWidth										(Public)
//
// Set the number of pixels to average to the left and right for smoothing.
////////////////////////////////////////////////////////////////////////////////
SPECTROVISSDK_INTERFACE_DECL void Spec_SetBoxcarWidth(SpecDevice* device, int boxcar);

////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION: Spec_GetBoxcarWidth										(Public)
//
// Get the number of pixels to average to the left and right for smoothing.
////////////////////////////////////////////////////////////////////////////////
SPECTROVISSDK_INTERFACE_DECL int  Spec_GetBoxcarWidth(SpecDevice* device);

////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION: Spec_SetIntegrationTime									(Public)
//
// Set the number of microseconds to "leave shutter open" for each scan.
// IntegrationTime = SampleTime
////////////////////////////////////////////////////////////////////////////////
SPECTROVISSDK_INTERFACE_DECL SpecStatus Spec_SetIntegrationTime(SpecDevice* device, unsigned long usecs);

////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION: Spec_GetIntegrationTime									(Public)
//
// Get the number of microseconds to "leave shutter open" for each scan.
// IntegrationTime = SampleTime
////////////////////////////////////////////////////////////////////////////////
SPECTROVISSDK_INTERFACE_DECL int  Spec_GetIntegrationTime(SpecDevice* device);	// returns micro seconds value

////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION: Spec_GetNumberOfPixels									(Public)
//
// Return the number of effective pixels for this device.
// We ASSUME here that the pixel count is fixed by the device type
// AND that we already know the device type (ProductID) in the descriptor
SPECTROVISSDK_INTERFACE_DECL int  Spec_GetNumberOfPixels(SpecDevice* device);

////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION: Spec_GetMaximumIntensity										(Public)
//
// Return the maximum intensity value for the pixels on this device.
// We ASSUME here that the maximum intensity is fixed by the device type
// AND that we already know the device type (ProductID) in the descriptor
////////////////////////////////////////////////////////////////////////////////
SPECTROVISSDK_INTERFACE_DECL int  Spec_GetMaximumIntensity(SpecDevice* device);

////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION: Spec_GetMinSampleTime										(Public)
//
// Return the smallest number of ms that can be used for sample time (aka integration time) 
// We ASSUME here that this value is fixed by the device type
// AND that we already know the device type (ProductID) in the descriptor
////////////////////////////////////////////////////////////////////////////////
SPECTROVISSDK_INTERFACE_DECL int  Spec_GetMinSampleTime(SpecDevice* device); // RETURN the smallest value (in ms) that this device can use for sample time (aka integration time)

////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION: Spec_GetMaxSampleTime										(Public)
//
// Return the Largest number of ms that can be used for sample time (aka integration time) 
// We ASSUME here that the this value is fixed by the device type
// AND that we already know the device type (ProductID) in the descriptor
////////////////////////////////////////////////////////////////////////////////
SPECTROVISSDK_INTERFACE_DECL int  Spec_GetMaxSampleTime(SpecDevice* device); // RETURN the largest value (in ms) this device can use

////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION: Spec_GetCalculatedDarkMaxCount										(Public)
//
// Return the maximum intensity value for the pixels on this device.
// AFTER ALLOWING FOR THE LAST Dark Pixel Correction!
// 
////////////////////////////////////////////////////////////////////////////////
SPECTROVISSDK_INTERFACE_DECL double Spec_GetCalculatedDarkMaxCount(SpecDevice* device);

// only used on Mac OS X
SPECTROVISSDK_INTERFACE_DECL SpecStatus Spec_GetSerialNumberByLocation(UInt32 location, char* outSNBuf);


#endif